<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class LoginSlugController extends Controller
{
    /**
     * Generate a new random slug and redirect the user there.
     * Example: https://boon-pharma.com/AbC123... -> shows login if matches.
     */
    public function redirectToRandom(Request $request)
    {
        $length = (int) config('app.login_slug_length', 40); // default 40 chars
        $ttl    = (int) config('app.login_slug_ttl', 15);    // default 15 minutes

        // Generate a URL-safe random slug (alphanumeric)
        $slug = Str::random($length);

        // Store current valid slug with TTL (minutes)
        $expiresAt = now()->addMinutes($ttl);
        try {
            Cache::put('current_login_slug', $slug, $expiresAt);
        } catch (\Throwable $e) {
            // swallow cache errors; we'll fall back to session
        }

        // Also store in session as a fallback in case cache store is misconfigured
        $request->session()->put('current_login_slug', $slug);
        $request->session()->put('current_login_slug_expires_at', $expiresAt->timestamp);

        // Optional: single-use (invalidate after first view)
        // Cache::put('login_slug_uses', 1, now()->addMinutes($ttl));

        // Redirect using app base URL (respects subfolder like /test)
        return redirect(url('/' . $slug));
    }

    /**
     * Serve the login page only if the slug matches the cached one.
     */
    public function serveBySlug(Request $request, string $slug)
    {
        $current = null;
        try {
            $current = Cache::get('current_login_slug');
        } catch (\Throwable $e) {
            // ignore cache read errors
        }

        // Fallback to session if cache is empty or unavailable
        if (!$current) {
            $sessionSlug = $request->session()->get('current_login_slug');
            $sessionExp  = (int) $request->session()->get('current_login_slug_expires_at', 0);
            if ($sessionSlug && $sessionExp > time()) {
                $current = $sessionSlug;
            }
        }

        if ($current && hash_equals($current, $slug)) {
            // Optional single-use logic:
            // if (Cache::decrement('login_slug_uses') < 0) { abort(404); }

            // Return your normal login view/route handler here.
            // If you already use Laravel Breeze/Fortify/UI, you can return the view directly:
            return response()->view('auth.login');

            // Or, if your login is handled by another controller/route,
            // you could forward/return that instead.
        }

        abort(404);
    }
}
